package furny.ga.operators;

import furny.entities.Furniture;
import furny.ga.FurnEntry;
import furny.ga.FurnEntryList;
import furny.ga.FurnLayoutIndividual;
import furny.ga.PseudoSpace;
import furny.ga.RoomVector;
import furny.ga.util.CollectionUtils;
import furny.ga.util.FurnLayoutIOUtil;
import ga.core.GA;
import ga.core.goperators.ICrossoverOp;
import ga.core.goperators.ProbabilityOp;
import ga.core.individual.IndividualList;
import ga.core.validation.GAContext;

/**
 * Crossover operator that does the segment cut crossover. The function is
 * explained in the master thesis of Stephan Dreyer.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class SegmentCutCrossoverOp extends ProbabilityOp implements
    ICrossoverOp<FurnLayoutIndividual> {

  /**
   * Creates the operator with a given crossover probability.
   * 
   * @param pCrossOver
   *          Crossover probability.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public SegmentCutCrossoverOp(final int pCrossOver) {
    super(pCrossOver);
  }

  @Override
  public IndividualList<FurnLayoutIndividual> crossover(
      final FurnLayoutIndividual individual1,
      final FurnLayoutIndividual individual2, final GAContext context) {
    final IndividualList<FurnLayoutIndividual> list = new IndividualList<FurnLayoutIndividual>();

    // clone to keep the context
    final FurnLayoutIndividual ind1 = individual1.clone();
    final FurnLayoutIndividual ind2 = individual2.clone();

    if (doOperate() && !individual1.equals(individual2)) {
      // do the intersection between both genotypes
      final FurnEntryList cIntersection = new FurnEntryList(
          individual1.getFurnitures());
      cIntersection.retainAll(individual2.getFurnitures());

      // cA = c1 - cIntersect
      final FurnEntryList cA = new FurnEntryList(individual1.getFurnitures());
      cA.removeAll(cIntersection);

      // cB = c2 - cIntersect
      final FurnEntryList cB = new FurnEntryList(individual2.getFurnitures());
      cB.removeAll(cIntersection);

      // random order for both lists
      CollectionUtils.randomOrder(cA);
      CollectionUtils.randomOrder(cB);

      // define bigger and smaller list
      final FurnEntryList bigger = cA.size() >= cB.size() ? cA : cB;
      final FurnEntryList smaller = cA.size() < cB.size() ? cA : cB;

      // this is a copy of the bigger list that will not be modified
      final FurnEntryList biggerSource = new FurnEntryList(bigger);

      // get the max length
      final int lMax = biggerSource.size();

      // create a random cut index
      final int r = getRandom().nextInt(Math.max(lMax - 1, 1));

      for (int i = 0; i < lMax; i++) {
        // first segment(guaranteed before r)
        final FurnEntry sA = biggerSource.get(i);

        // optional segment
        final FurnEntry sB = smaller.size() > i ? smaller.get(i) : null;

        if (i <= r) {
          // before the cut do Uniform crossover
          if (sB != null) {
            if (getRandom().nextBoolean()) {
              // swap furniture
              final Furniture tmp = sA.getFurniture();
              sA.setFurniture(sB.getFurniture());
              sB.setFurniture(tmp);
            }

            int x1 = sA.getVector().getXGene();
            int x2 = sB.getVector().getXGene();

            if (getRandom().nextBoolean()) {
              // swap x
              final int tmp = x1;
              x1 = x2;
              x2 = tmp;
            }

            int y1 = sA.getVector().getYGene();
            int y2 = sB.getVector().getYGene();

            if (getRandom().nextBoolean()) {
              // swap y
              final int tmp = y1;
              y1 = y2;
              y2 = tmp;
            }

            int rot1 = sA.getVector().getRotationSteps();
            int rot2 = sB.getVector().getRotationSteps();

            if (getRandom().nextBoolean()) {
              // swap rot
              final int tmp = rot1;
              rot1 = rot2;
              rot2 = tmp;
            }

            // store the new vectors
            sA.setVector(new RoomVector(x1, y1, rot1));
            sB.setVector(new RoomVector(x2, y2, rot2));
          }
        } else {
          // after the cut swap segments

          if (sB != null) {
            // simple swap
            bigger.set(i, sB);
            smaller.set(i, sA);
          } else {
            smaller.add(sA);
            bigger.remove(sA);
          }
        }
      }

      // create the new individuals and add them to the list
      ind1.getFurnitures().clear();
      ind1.getFurnitures().addAll(cIntersection);
      ind1.getFurnitures().addAll(cA);

      ind2.getFurnitures().clear();

      ind2.getFurnitures().addAll(cIntersection);
      ind2.getFurnitures().addAll(cB);
    }

    // add the invididuals to the list
    list.add(ind1);
    list.add(ind2);

    return list;
  }

  /**
   * Main method for testing.
   * 
   * @param args
   *          No arguments required.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static void main(final String[] args) {
    final GAContext context = new GAContext();
    context.put(GA.KEY_VALIDATION_SPACE, new PseudoSpace(10f, 10f));

    final FurnLayoutIndividual ind1 = new FurnLayoutIndividual(context);
    ind1.initRandomly();

    final FurnLayoutIndividual ind2 = new FurnLayoutIndividual(context);
    ind2.initRandomly();
    ind2.getFurnitures().add(ind1.getFurnitures().get(0));

    final ICrossoverOp<FurnLayoutIndividual> crossover = new SegmentCutCrossoverOp(
        100);

    System.out.println("\n PARENT 1\n");
    System.out.println(FurnLayoutIOUtil.printSimpleGenotype(ind1
        .getSimpleGenotype()));

    System.out.println("\n PARENT 2\n");
    System.out.println(FurnLayoutIOUtil.printSimpleGenotype(ind2
        .getSimpleGenotype()));

    final IndividualList<FurnLayoutIndividual> list = crossover.crossover(ind1,
        ind2, context);

    for (int i = 0; i < list.size(); i++) {
      final FurnLayoutIndividual newInd = list.get(i);

      System.out.println("\n CHILD " + (i + 1) + "\n");
      System.out.println(FurnLayoutIOUtil.printSimpleGenotype(newInd
          .getSimpleGenotype()));
    }

  }
}
